import Events from 'events';
const emitter = new Events();

class Schedule {
    // Массив для хранения расписания
    schedule: number[];
    // Номер текущего периода в расписании
    currentPeriod: number;
    // Интервал для управления таймером
    interval?: NodeJS.Timeout;

    constructor(start: number, end: number, interval: number) {
        // Генерируем массив периодов на основе заданных параметров
        this.schedule = Array.from(
            { length: (end - start) / interval + 1 },
            (_, i) => start + interval * i
        );
        // Определяем начальное значение текущего периода
        this.currentPeriod = this.findCurrentPeriod();
    }

    // Метод для определения текущего периода
    findCurrentPeriod() {
        let now = Date.now();
        // Проходим по расписанию и ищем ближайший период
        for (let i = 1; i < this.schedule.length; i++) {
            if (this.schedule[i] >= now) {
                return i - 1;
            }
        }
        // Если текущее время за пределами расписания
        return -1;
    }

    // Метод для запуска таймера
    start() {
        // console.log(
        //     this.schedule.map((el) =>
        //         new Date(el).toLocaleTimeString('en-US', {
        //             timeZone: 'Europe/Moscow',
        //             hour12: false,
        //         })
        //     )
        // );
        // Устанавливаем интервальный таймер
        this.interval = setInterval(() => {
            // Определяем текущий период
            const newPeriod = this.findCurrentPeriod();
            // Если расписание завершено
            if (newPeriod === -1) {
                // Генерируем событие "schedule-end"

                emitter.emit('schedule-end');
                // Останавливаем таймер
                this.stop();
                return;
            }
            // Если произошло изменение периода
            if (newPeriod !== this.currentPeriod) {
                // Обновляем текущий период
                this.currentPeriod = newPeriod;
                // Генерируем событие "interval" с информацией о новом интервале
                emitter.emit(
                    'interval',
                    'new interval: ' +
                        new Date(
                            this.schedule[this.currentPeriod]
                        ).toISOString()
                );
            }
        }, 1000);
    }
    // Метод для остановки таймера
    stop() {
        // Очищаем интервал таймера
        clearInterval(this.interval);
    }

    // Метод для подписки на событие
    on(name: string, action: (...data: any[]) => void) {
        // Добавляем обработчик события
        emitter.on(name, action);
    }
}

export { Schedule };
